#!/bin/bash

# Quick Test Script - Verifies all components before running full script

set -e

GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m'

print_success() { echo -e "${GREEN}✅ $1${NC}"; }
print_warning() { echo -e "${YELLOW}⚠️  $1${NC}"; }
print_error() { echo -e "${RED}❌ $1${NC}"; }

echo "=========================================="
echo "Quick Pre-Flight Test"
echo "=========================================="
echo ""

# Test 1: Python and virtual environment
echo "Test 1: Python environment..."
if [ -d "venv" ]; then
    source venv/bin/activate
    if python3 -c "import llama_index.core; import faiss; import reportlab" 2>/dev/null; then
        print_success "Python dependencies OK"
    else
        print_error "Missing Python dependencies. Run: pip install -r requirements.txt"
        exit 1
    fi
else
    print_error "Virtual environment not found. Run: ./setup_server.sh"
    exit 1
fi

# Test 2: Ollama connection
echo ""
echo "Test 2: Ollama service..."
if curl -s http://localhost:11434/api/tags > /dev/null 2>&1; then
    print_success "Ollama service is running"
else
    print_error "Ollama service not running. Start with: ollama serve"
    exit 1
fi

# Test 3: Model availability
echo ""
echo "Test 3: Model availability..."
if ollama list | grep -q "llama3.1:8b"; then
    print_success "Model llama3.1:8b is available"
else
    print_error "Model not found. Install with: ollama pull llama3.1:8b"
    exit 1
fi

# Test 4: Model response
echo ""
echo "Test 4: Model response test..."
RESPONSE=$(ollama run llama3.1:8b "Say OK if you can read this." 2>&1 | head -1)
if [ ! -z "$RESPONSE" ]; then
    print_success "Model is responding"
else
    print_warning "Model response test inconclusive"
fi

# Test 5: Data directory
echo ""
echo "Test 5: Data directory..."
if [ -d "Niraj_Upload" ]; then
    FILE_COUNT=$(find Niraj_Upload -type f 2>/dev/null | wc -l)
    if [ "$FILE_COUNT" -gt 0 ]; then
        print_success "Data directory has $FILE_COUNT file(s)"
    else
        print_warning "Data directory is empty"
    fi
else
    print_warning "Data directory not found (will be created)"
fi

# Test 6: Script syntax
echo ""
echo "Test 6: Script syntax..."
if python3 -m py_compile financial_report_gen.py 2>/dev/null; then
    print_success "Script syntax is valid"
else
    print_error "Script has syntax errors"
    exit 1
fi

echo ""
echo "=========================================="
echo "All pre-flight checks passed! ✅"
echo "=========================================="
echo ""
echo "You can now run the full script:"
echo "  python3 financial_report_gen.py"
echo ""



