from llama_index.core import SimpleDirectoryReader, VectorStoreIndex, Settings, StorageContext
from llama_index.llms.ollama import Ollama
from llama_index.embeddings.huggingface import HuggingFaceEmbedding
from llama_index.vector_stores.faiss import FaissVectorStore
import os
import faiss
from reportlab.platypus import SimpleDocTemplate, Paragraph, Spacer, PageBreak, Table, TableStyle
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.units import inch
from reportlab.lib.colors import HexColor
from reportlab.lib.pagesizes import letter
from reportlab.pdfgen.canvas import Canvas
from datetime import datetime
import re

# 1. Load all financial data from folder (recursively reads all subdirectories)
print("📂 Loading documents from Niraj_Upload folder...")
docs = SimpleDirectoryReader("Niraj_Upload", recursive=True).load_data()
print(f"✅ Loaded {len(docs)} documents successfully")

# 2. Connect to local Llama 3 model running via Ollama
# Increased timeout for comprehensive report generation (30 minutes)
# Llama 3 8B provides better reasoning capabilities for complex financial analysis
# Note: Use 'llama3:8b' (the base model supports instruction-following)
llm = Ollama(model="llama3:8b", request_timeout=1800)
Settings.llm = llm

# 3. Use local embedding model (no API key required)
embed_model = HuggingFaceEmbedding(model_name="BAAI/bge-small-en-v1.5")
Settings.embed_model = embed_model

# 4. Create FAISS vector index (enables fast chunking and search for large datasets)
# FAISS provides faster similarity search and persistent storage for 100+ MB datasets
faiss_index_path = "./faiss_index"
print("📊 Creating FAISS vector index from documents...")

# Check if index already exists
index_exists = os.path.exists(faiss_index_path) and os.path.isdir(faiss_index_path)

if index_exists:
    print(f"   📂 Loading existing FAISS index from {faiss_index_path}...")
    try:
        # Load FAISS vector store explicitly
        faiss_vector_store = FaissVectorStore.from_persist_dir(faiss_index_path)
        # Load the storage context with the FAISS vector store and other persisted components
        storage_context = StorageContext.from_defaults(
            vector_store=faiss_vector_store,
            persist_dir=faiss_index_path
        )
        
        # Try to rebuild index structure - get all nodes from docstore
        all_nodes = []
        if storage_context.docstore:
            # Get all nodes from docstore
            print(f"   🔄 Rebuilding index structure from docstore...")
            for doc_id in list(storage_context.docstore.docs.keys()):
                try:
                    # Try to get node by doc_id
                    doc_ref = storage_context.docstore.docs.get(doc_id)
                    if doc_ref:
                        # Get all nodes associated with this doc
                        if hasattr(doc_ref, 'node_id'):
                            node = storage_context.docstore.get_node(doc_ref.node_id)
                            if node:
                                all_nodes.append(node)
                        # Also try getting by doc_id directly as node_id
                        try:
                            node = storage_context.docstore.get_node(doc_id)
                            if node and node not in all_nodes:
                                all_nodes.append(node)
                        except:
                            pass
                except Exception as e:
                    continue
        
        # If we have nodes, try to rebuild the index
        if all_nodes and len(all_nodes) > 0:
            print(f"   ✅ Found {len(all_nodes)} nodes, rebuilding index structure...")
            # Create index with nodes - this should rebuild the index structure
            index = VectorStoreIndex(
                nodes=all_nodes,
                storage_context=storage_context
            )
            
            # Verify the index structure is properly synced with vector store
            # Check if index structure has nodes and matches vector store
            if hasattr(index, 'index_struct') and hasattr(index.index_struct, 'nodes_dict'):
                nodes_in_struct = len(index.index_struct.nodes_dict)
                print(f"   ✅ Index structure rebuilt with {nodes_in_struct} nodes")
                
                # Check if vector store has entries
                vector_store_size = 0
                if hasattr(faiss_vector_store, '_faiss_index'):
                    vector_store_size = faiss_vector_store._faiss_index.ntotal
                
                # If there's a mismatch, the index is corrupted
                if nodes_in_struct == 0:
                    raise ValueError("Index structure is empty after rebuild")
                if vector_store_size > 0 and nodes_in_struct < vector_store_size * 0.5:
                    # Significant mismatch - index is likely corrupted
                    raise ValueError(f"Index structure mismatch: {nodes_in_struct} nodes vs {vector_store_size} vectors")
            else:
                raise ValueError("Index structure not properly initialized")
            
            print("✅ Loaded existing FAISS index successfully")
        else:
            raise ValueError("No nodes found in docstore")
            
    except Exception as e:
        print(f"   ⚠️  Error loading persisted index: {e}")
        print(f"   🔨 Rebuilding index from documents (this may take a few minutes)...")
        # Delete the corrupted index and rebuild
        import shutil
        if os.path.exists(faiss_index_path):
            shutil.rmtree(faiss_index_path)
        index_exists = False

# Build new index if it doesn't exist or if loading failed
if not index_exists:
    print(f"   🔨 Building new FAISS index (this may take a few minutes)...")
    # Get embedding dimension from the embedding model
    # BAAI/bge-small-en-v1.5 has dimension 384
    embedding_dim = 384
    
    # Create a new FAISS index (using L2 distance for similarity search)
    faiss_index = faiss.IndexFlatL2(embedding_dim)
    
    # Wrap it in FaissVectorStore
    faiss_vector_store = FaissVectorStore(faiss_index=faiss_index)
    storage_context = StorageContext.from_defaults(vector_store=faiss_vector_store)
    
    # Create index from documents
    index = VectorStoreIndex.from_documents(
        docs,
        storage_context=storage_context,
        show_progress=True
    )
    
    # Persist the storage context (which includes the FAISS vector store)
    storage_context.persist(persist_dir=faiss_index_path)
    print(f"✅ FAISS index created and saved to {faiss_index_path}")

# Use tree_summarize for faster processing of large documents
# This is more efficient than refine for long responses
from llama_index.core.response_synthesizers import TreeSummarize

print("🔍 Setting up query engine for in-depth analysis...")
# Note: similarity_top_k reduced slightly to avoid index mismatch issues
# The index will be rebuilt properly on first run
query_engine = index.as_query_engine(
    similarity_top_k=20,  # Balanced for comprehensive coverage while avoiding index issues
    response_mode="tree_summarize",  # Efficient for long responses
    verbose=True
)
print("✅ Query engine ready")

# 5. Ask it to produce a comprehensive, professional financial report
prompt = """
You are a senior financial analyst at a top-tier investment bank (Goldman Sachs, Morgan Stanley, or JPMorgan level). 
Based on ALL the provided documents (including quarterly reports, annual reports, SEC filings, press releases, 
dividend history, capital return history, green bond reports, and net sales data), generate an EXHAUSTIVE, 
deep-dive, professional-grade financial analysis report.

The report must be EXTREMELY comprehensive, detailed, analytical, and suitable for C-suite executives, 
institutional investors, and major bank analysts. This is not a summary - it is a full investment-grade 
equity research report that rivals reports from top-tier investment banks.

CRITICAL REQUIREMENTS:
- Provide DEEP analysis, not surface-level summaries
- Include extensive quantitative data with specific numbers, percentages, and calculations
- Show detailed year-over-year and quarter-over-quarter comparisons with percentage changes
- Include multi-year trend analysis with compound annual growth rates (CAGR)
- Provide detailed explanations of WHY trends occurred, not just WHAT happened
- Include forward-looking analysis and projections based on historical patterns
- Cross-reference data across multiple documents to provide comprehensive insights
- Include detailed financial calculations and ratio analysis
- Provide context and industry comparisons where possible
- Each section should be substantial (minimum 2-3 pages of detailed analysis per major section)

REQUIRED SECTIONS (provide detailed analysis for each):

1. EXECUTIVE SUMMARY (Minimum 3-4 pages)
   - Comprehensive company overview and strategic positioning
   - Detailed key financial metrics and performance highlights with specific numbers
   - Major achievements and milestones with dates and impact
   - Investment thesis summary with supporting evidence
   - Key investment highlights and value proposition
   - Summary of financial performance across all periods
   - Strategic direction and management vision

2. FINANCIAL PERFORMANCE ANALYSIS (Minimum 8-10 pages)
   - Detailed revenue trends and growth analysis (quarterly and annual) with specific dollar amounts and percentages
   - Comprehensive profitability metrics analysis:
     * Gross margin trends with detailed breakdown by segment
     * Operating margin analysis with operating expense trends
     * Net margin analysis with tax impact
     * Margin expansion/contraction drivers and explanations
   - In-depth Earnings per Share (EPS) analysis:
     * Basic and diluted EPS trends
     * EPS growth rates and drivers
     * Share count impact on EPS
   - Comprehensive cash flow analysis:
     * Operating cash flow trends with detailed drivers
     * Investing cash flow with capital expenditure analysis
     * Financing cash flow including debt and equity activities
     * Free cash flow generation and trends
     * Cash conversion cycle analysis
   - Detailed balance sheet strength and liquidity position:
     * Current assets and liabilities analysis
     * Cash and cash equivalents trends
     * Short-term and long-term debt analysis
     * Equity analysis and retained earnings
   - Working capital management:
     * Accounts receivable trends
     * Inventory management and turnover
     * Accounts payable analysis
     * Working capital efficiency metrics
   - Comprehensive debt analysis and capital structure:
     * Debt-to-equity ratios over time
     * Interest coverage ratios
     * Debt maturity schedule analysis
     * Credit rating considerations

3. FINANCIAL RATIOS AND METRICS (Minimum 5-6 pages)
   - Detailed profitability ratios analysis:
     * Return on Equity (ROE) with DuPont analysis breakdown
     * Return on Assets (ROA) trends and drivers
     * Return on Invested Capital (ROIC) with detailed calculations
     * Gross profit margin, operating profit margin, net profit margin trends
     * Year-over-year and multi-year comparisons
   - Comprehensive efficiency ratios:
     * Asset turnover ratios by asset category
     * Inventory turnover with days inventory outstanding
     * Accounts receivable turnover with days sales outstanding
     * Total asset utilization efficiency
   - Detailed liquidity ratios:
     * Current ratio trends and analysis
     * Quick ratio (acid-test) analysis
     * Cash ratio analysis
     * Operating cash flow ratios
   - In-depth leverage ratios:
     * Debt-to-equity ratio trends
     * Debt-to-assets ratio analysis
     * Interest coverage ratio with detailed calculations
     * Equity multiplier analysis
   - Comprehensive valuation metrics (where market data available):
     * Price-to-Earnings (P/E) ratio analysis
     * Price-to-Book (P/B) ratio trends
     * Enterprise Value to EBITDA (EV/EBITDA)
     * Price-to-Sales ratio
   - Multi-period comparative analysis with trend identification

4. SEGMENT AND PRODUCT ANALYSIS (Minimum 6-8 pages)
   - Detailed revenue breakdown by product category:
     * iPhone revenue trends with unit sales and ASP analysis
     * Mac revenue and market positioning
     * iPad revenue trends and market dynamics
     * Services revenue with detailed breakdown by service type
     * Wearables, Home and Accessories revenue analysis
     * Other products revenue
     * Year-over-year growth rates for each category
     * Product mix evolution over time
   - Comprehensive geographic revenue distribution:
     * Americas revenue trends and drivers
     * Europe revenue analysis
     * Greater China revenue with detailed insights
     * Japan revenue trends
     * Rest of Asia Pacific revenue
     * Currency impact analysis
     * Regional growth rates and market dynamics
   - In-depth segment profitability analysis:
     * Gross margin by product category
     * Operating margin by segment
     * Segment contribution to overall profitability
     * Margin trends and drivers by segment
   - Detailed market share and competitive positioning:
     * Market share trends in key product categories
     * Competitive positioning vs. major competitors
     * Market dynamics and industry trends
   - Comprehensive product lifecycle and innovation impact:
     * New product launches and their financial impact
     * Product refresh cycles and revenue impact
     * Innovation investments and returns
     * Technology trends and competitive advantages

5. QUARTERLY PERFORMANCE TRENDS (2022-2025) (Minimum 10-12 pages)
   - Detailed Q1, Q2, Q3, Q4 analysis for each fiscal year:
     * Revenue analysis for each quarter with specific numbers
     * Profitability trends quarter-by-quarter
     * Key product launches and their impact by quarter
     * Seasonal factors affecting each quarter
   - Comprehensive seasonal patterns and cyclicality analysis:
     * Holiday season impact (Q1 fiscal year)
     * Back-to-school season effects
     * Product launch timing and impact
     * Historical seasonal patterns
   - In-depth year-over-year and quarter-over-quarter comparisons:
     * YoY revenue growth rates for each quarter
     * QoQ sequential growth/decline analysis
     * Margin trends YoY and QoQ
     * EPS growth comparisons
   - Detailed analysis of key drivers of quarterly variations:
     * Product launch timing impact
     * Economic factors affecting quarters
     * Supply chain and operational factors
     * Market conditions and competitive dynamics
     * Currency exchange rate impacts

6. ANNUAL PERFORMANCE REVIEW (Minimum 8-10 pages)
   - Comprehensive fiscal year analysis for 2022, 2023, 2024, 2025:
     * Full-year revenue with detailed breakdown
     * Annual profitability metrics and margins
     * Key financial highlights for each year
     * Major events and milestones per year
     * Strategic initiatives launched each year
   - In-depth multi-year trend analysis:
     * 4-year revenue CAGR
     * Profitability trend analysis over 4 years
     * Cash flow generation trends
     * Balance sheet evolution
     * Market capitalization trends (if available)
   - Detailed annual growth rates and compound metrics:
     * Revenue CAGR calculations
     * EPS CAGR analysis
     * Free cash flow growth rates
     * Return metrics trends
     * Market share evolution

7. CAPITAL ALLOCATION AND SHAREHOLDER RETURNS
   - Dividend history and policy analysis
   - Share buyback programs and capital return timeline
   - Dividend yield and payout ratio trends
   - Total shareholder return analysis

8. ENVIRONMENTAL, SOCIAL, AND GOVERNANCE (ESG) ANALYSIS
   - Green bond initiatives and environmental progress
   - Sustainability metrics and commitments
   - ESG risk assessment
   - Corporate responsibility initiatives

9. MANAGEMENT DISCUSSION AND ANALYSIS (MD&A)
   - Management's perspective on financial results
   - Strategic initiatives and business priorities
   - Risk factors and mitigation strategies
   - Operational highlights and challenges

10. COMPETITIVE LANDSCAPE AND MARKET POSITION
    - Industry positioning
    - Competitive advantages and moats
    - Market share dynamics
    - Competitive threats and opportunities

11. PRESS RELEASE INSIGHTS AND MANAGEMENT COMMENTARY
    - Key quotes and management statements from quarterly releases
    - Strategic messaging and priorities
    - Forward-looking statements analysis

12. RISK ANALYSIS
    - Financial risks
    - Operational risks
    - Market and competitive risks
    - Regulatory and geopolitical risks
    - Technology and innovation risks

13. FUTURE OUTLOOK AND FORECASTING
    - Growth prospects and opportunities
    - Strategic initiatives and investments
    - Market trends and industry outlook
    - Potential challenges and headwinds
    - Long-term value creation potential

14. INVESTMENT RECOMMENDATIONS AND CONCLUSIONS
    - Investment thesis summary
    - Key investment highlights
    - Risk-reward assessment
    - Target audience considerations

FORMATTING REQUIREMENTS:
- Start with a detailed TABLE OF CONTENTS listing all sections and subsections
- Use clear section numbering (1., 2., 3., etc.) and subsection numbering (1.1, 1.2, etc.)
- Include SPECIFIC numbers, percentages, dollar amounts, and data points throughout - never use vague terms like "increased" without the specific percentage
- Use professional financial terminology and industry-standard metrics
- Provide DEEP analysis with explanations, not just summaries or data dumps
- Include detailed comparative analysis (YoY, QoQ, multi-year trends)
- Include calculations and show your work (e.g., "Revenue grew 8.5% from $394.3B to $428.0B")
- Each major section should be substantial (minimum 3-5 pages of detailed analysis)
- Include tables and structured data presentations where helpful
- Make the report EXTENSIVE and DETAILED - aim for 80-100+ pages of comprehensive content
- Every claim should be supported by specific data from the documents
- Provide context and explanations for all trends and changes

DEPTH REQUIREMENTS:
- Don't just state facts - analyze and explain them
- Show relationships between different metrics
- Provide forward-looking insights based on historical patterns
- Include detailed breakdowns and sub-analyses
- Cross-reference information across multiple documents
- Provide detailed explanations of drivers and causes

TONE: Professional, analytical, comprehensive, authoritative - suitable for institutional investors, C-suite executives, and top-tier investment bank research standards.

Generate the report now with ALL sections fully detailed, in-depth, and comprehensive. This should be a publication-quality equity research report.
"""

# 6. Generate the comprehensive report
print("\n" + "="*70)
print("🚀 Starting in-depth report generation...")
print("⏱️  This may take 30-60 minutes due to the comprehensive, deep-dive analysis")
print("📊 Generating 80-100+ page publication-quality equity research report...")
print("="*70 + "\n")

try:
    result = query_engine.query(prompt)
    print("\n" + "="*70)
    print("✅ Report generation completed!")
    print("="*70 + "\n")
    print(f"📄 Report length: {len(result.response)} characters")
    print(f"📝 Estimated pages: ~{len(result.response) // 2000} pages\n")
except KeyError as e:
    # Handle index structure mismatch - rebuild index
    # This happens when vector store has node IDs that aren't in index structure
    print(f"\n⚠️  Index structure mismatch detected: {e}")
    print("🔄 Automatically rebuilding index from scratch...")
    import shutil
    if os.path.exists(faiss_index_path):
        shutil.rmtree(faiss_index_path)
    
    # Rebuild index
    print("   🔨 Building new FAISS index (this may take a few minutes)...")
    embedding_dim = 384
    faiss_index = faiss.IndexFlatL2(embedding_dim)
    faiss_vector_store = FaissVectorStore(faiss_index=faiss_index)
    storage_context = StorageContext.from_defaults(vector_store=faiss_vector_store)
    
    index = VectorStoreIndex.from_documents(
        docs,
        storage_context=storage_context,
        show_progress=True
    )
    storage_context.persist(persist_dir=faiss_index_path)
    print("   ✅ Index rebuilt successfully")
    
    # Recreate query engine with new index
    query_engine = index.as_query_engine(
        similarity_top_k=20,
        response_mode="tree_summarize",
        verbose=True
    )
    
    # Retry query
    print("   🔄 Retrying report generation...")
    result = query_engine.query(prompt)
    print("\n" + "="*70)
    print("✅ Report generation completed!")
    print("="*70 + "\n")
    print(f"📄 Report length: {len(result.response)} characters")
    print(f"📝 Estimated pages: ~{len(result.response) // 2000} pages\n")
except Exception as e:
    print(f"\n❌ Error during report generation: {e}")
    print("\n💡 Suggestions:")
    print("   - Try increasing request_timeout further")
    print("   - Check if Ollama is running: ollama list")
    print("   - Consider using a faster model or reducing similarity_top_k")
    raise

print("📋 Report preview (first 500 characters):")
print("-" * 70)
print(result.response[:500] + "...\n")

# === START OF make_pdf FUNCTION CHANGES ===

# New function for drawing the branded first page (cover page styling)
def first_page(canvas, doc):
    canvas.saveState()
    # Draw a prominent vertical color strip on the left (simulating a brand spine)
    canvas.setFillColor(HexColor('#002D58')) # Deep Navy Blue
    canvas.rect(0, 0, 15, letter[1], fill=1, stroke=0) # 15-point wide rectangle on the left edge
    
    # Draw a colored rule beneath the main title area
    canvas.setFillColor(HexColor('#1a1a1a'))
    canvas.rect(72, letter[1] - 170, letter[0] - 144, 2, fill=1, stroke=0)
    canvas.restoreState()


def make_pdf(text, filename="financial_report.pdf"):
    # Custom canvas for headers and footers
    class NumberedCanvas(Canvas):
        def __init__(self, *args, **kwargs):
            Canvas.__init__(self, *args, **kwargs)
            self._saved_page_states = []
        
        def showPage(self):
            self._saved_page_states.append(dict(self.__dict__))
            self._startPage()
        
        def save(self):
            page_count = len(self._saved_page_states)
            for state in self._saved_page_states:
                self.__dict__.update(state)
                self.draw_page_number(page_count)
                Canvas.showPage(self)
            Canvas.save(self)
        
        # --- MODIFIED draw_page_number for Branded Header/Footer ---
        def draw_page_number(self, page_count):
            self.saveState()
            
            # --- HEADER CONTENT ---
            self.setFont("Helvetica-Bold", 9)
            self.setFillColor(HexColor('#1a1a1a'))
            # Draw Firm/Report Name in Header (Top Left)
            self.drawString(72, letter[1] - 45, "Oxyfinz LLC | Equity Research")
            
            # Header line (moved down slightly)
            self.setStrokeColor(HexColor('#d0d0d0'))
            self.setLineWidth(0.5)
            self.line(72, letter[1] - 55, letter[0] - 72, letter[1] - 55) # Line lowered

            # --- FOOTER CONTENT ---
            self.setFont("Helvetica", 8)
            self.setFillColor(HexColor('#666666'))

            # Confidentiality Notice (Bottom Left)
            self.drawString(72, 30, "CONFIDENTIAL & PROPRIETARY")
            
            # Page number (Bottom Right)
            page_text = f"Page {self._pageNumber} of {page_count}"
            self.drawRightString(letter[0] - 72, 30, page_text)
            
            # Footer line (moved up slightly)
            self.line(72, 45, letter[0] - 72, 45)
            
            self.restoreState()
        # --- END OF MODIFIED draw_page_number ---

    
    doc = SimpleDocTemplate(
        filename,
        pagesize=letter,
        rightMargin=72,
        leftMargin=72,
        topMargin=90,
        bottomMargin=72
    )
    
    # Create custom styles
    styles = getSampleStyleSheet()
    
    # Title style
    title_style = ParagraphStyle(
        'CustomTitle',
        parent=styles['Heading1'],
        fontSize=24,
        textColor=HexColor('#1a1a1a'),
        spaceAfter=30,
        alignment=1,  # Center
        fontName='Helvetica-Bold'
    )
    
    # Section heading style (more professional)
    heading_style = ParagraphStyle(
        'CustomHeading',
        parent=styles['Heading1'],
        fontSize=18,
        textColor=HexColor('#1a1a1a'),
        spaceBefore=25,
        spaceAfter=15,
        fontName='Helvetica-Bold',
        borderColor=HexColor('#3498db'),
        borderWidth=2,
        borderPadding=8,
        backColor=HexColor('#f8f9fa'),
        leftIndent=0,
        rightIndent=0
    )
    
    # Subsection heading style
    subheading_style = ParagraphStyle(
        'CustomSubHeading',
        parent=styles['Heading2'],
        fontSize=14,
        textColor=HexColor('#34495e'),
        spaceBefore=15,
        spaceAfter=10,
        fontName='Helvetica-Bold'
    )
    
    # --- MODIFIED Body text style (Using Times-Roman for long-form readability) ---
    body_style = ParagraphStyle(
        'CustomBody',
        parent=styles['Normal'],
        fontSize=10.5,
        textColor=HexColor('#2c2c2c'),
        spaceAfter=10,
        leading=15,
        fontName='Times-Roman', # <--- Changed from 'Helvetica' to 'Times-Roman'
        alignment=4,  # Justify
        firstLineIndent=0
    )
    # --- END OF MODIFIED Body text style ---
    
    # Key metrics/highlight style
    highlight_style = ParagraphStyle(
        'Highlight',
        parent=styles['Normal'],
        fontSize=10.5,
        textColor=HexColor('#1a1a1a'),
        spaceAfter=8,
        leading=14,
        fontName='Helvetica-Bold',
        leftIndent=20,
        backColor=HexColor('#f0f4f8')
    )
    
    # TOC styles
    toc_title_style = ParagraphStyle(
        'TOCTitle',
        parent=styles['Heading1'],
        fontSize=18,
        textColor=HexColor('#2c3e50'),
        spaceAfter=20,
        alignment=1,
        fontName='Helvetica-Bold'
    )
    
    # **NEW TOC HEADER STYLE FOR TABLE**
    toc_header_style = ParagraphStyle(
        'TOCHeader',
        parent=styles['Normal'],
        fontSize=11,
        textColor=HexColor('#ffffff'), # Text color for header (matches TableStyle)
        alignment=1, # Center
        fontName='Helvetica-Bold'
    )
    
    toc_item_style = ParagraphStyle(
        'TOCItem',
        parent=styles['Normal'],
        fontSize=11,
        textColor=HexColor('#34495e'),
        spaceAfter=8,
        leftIndent=0,
        fontName='Helvetica'
    )
    
    toc_subitem_style = ParagraphStyle(
        'TOCSubItem',
        parent=styles['Normal'],
        fontSize=10,
        textColor=HexColor('#7f8c8d'),
        spaceAfter=6,
        leftIndent=20,
        fontName='Helvetica'
    )
    
    # Build content
    story = []
    toc_entries = []
    
    # Professional Cover Page (like analyst reports)
    title_page = []
    title_page.append(Spacer(1, 1.5*inch))
    
    # Company name
    company_style = ParagraphStyle(
        'CompanyName', parent=styles['Heading1'],
        fontSize=32, textColor=HexColor('#1a1a1a'),
        spaceAfter=10, alignment=1, fontName='Helvetica-Bold'
    )
    title_page.append(Paragraph("APPLE INC.", company_style))
    title_page.append(Spacer(1, 0.1*inch))
    title_page.append(Paragraph("NASDAQ: AAPL", 
                          ParagraphStyle('Ticker', parent=styles['Normal'], 
                                        fontSize=16, textColor=HexColor('#666666'),
                                        alignment=1, fontName='Helvetica')))
    
    title_page.append(Spacer(1, 0.8*inch))
    
    # Report title
    title_page.append(Paragraph("COMPREHENSIVE FINANCIAL ANALYSIS REPORT", title_style))
    title_page.append(Spacer(1, 0.3*inch))
    title_page.append(Paragraph("Investment-Grade Equity Research", 
                          ParagraphStyle('Subtitle', parent=styles['Normal'], 
                                        fontSize=16, textColor=HexColor('#2c3e50'),
                                        alignment=1, fontName='Helvetica-Bold')))
    
    title_page.append(Spacer(1, 1.2*inch))
    
    # Report metadata
    metadata_style = ParagraphStyle(
        'Metadata', parent=styles['Normal'],
        fontSize=11, textColor=HexColor('#555555'),
        alignment=1, fontName='Helvetica'
    )
    title_page.append(Paragraph(f"Report Date: {datetime.now().strftime('%B %d, %Y')}", metadata_style))
    title_page.append(Spacer(1, 0.1*inch))
    title_page.append(Paragraph("Analysis Period: Fiscal Years 2022-2025", metadata_style))
    title_page.append(Spacer(1, 0.1*inch))
    title_page.append(Paragraph("Report Type: Comprehensive Financial Analysis", metadata_style))
    
    title_page.append(Spacer(1, 1.5*inch))
    
    # Disclaimer
    disclaimer_style = ParagraphStyle(
        'Disclaimer', parent=styles['Normal'],
        fontSize=9, textColor=HexColor('#999999'),
        alignment=1, fontName='Helvetica',
        leading=11
    )
    title_page.append(Paragraph("This report is for informational purposes only and does not constitute", disclaimer_style))
    title_page.append(Paragraph("investment advice. All data and analysis are based on publicly available information.", disclaimer_style))
    
    title_page.append(PageBreak())
    
    # Process text content and extract TOC
    # First, try to split by double newlines, but also handle single newlines for better parsing
    paragraphs = []
    
    # Handle different newline formats
    text_normalized = text.replace('\r\n', '\n').replace('\r', '\n')
    
    for para in text_normalized.split("\n\n"):
        para = para.strip()
        if para:
            # Further split by single newlines if the paragraph is very long (might be multiple paragraphs)
            if len(para) > 500 and '\n' in para:
                for subpara in para.split('\n'):
                    subpara = subpara.strip()
                    if subpara and len(subpara) > 3:  # Ignore very short lines
                        paragraphs.append(subpara)
            else:
                if len(para) > 3:  # Ignore very short paragraphs
                    paragraphs.append(para)
    
    print(f"   📝 Processed {len(paragraphs)} paragraphs from report text")
    
    current_section = None
    current_subsection = None
    
    for para in paragraphs:
        if not para:
            continue
        
        # Check if it's a main heading (numbered section like "1. EXECUTIVE SUMMARY")
        main_heading_match = re.match(r'^(\d+)\.\s+([A-Z][A-Z\s&()]+)', para)
        if main_heading_match:
            section_num = main_heading_match.group(1)
            section_title = main_heading_match.group(2).strip()
            
            # --- CRITICAL CHANGE: Force a new page for every major section ---
            # Do not insert PageBreak before the first section (which follows TOC)
            if int(section_num) > 1:
                story.append(PageBreak())
            # --- END OF CRITICAL CHANGE ---

            # Store full title for TOC, without truncation
            toc_entries.append((section_num, para, None)) 
            current_section = section_num
            current_subsection = None
            
            # Add extra spacing before major sections
            story.append(Spacer(1, 0.2*inch))
            story.append(Paragraph(para, heading_style))
            story.append(Spacer(1, 0.1*inch))
        
        # Check if it's a subsection (numbered like 1.1, 1.2, etc.)
        elif re.match(r'^\d+\.\d+\.', para):
            if current_section:
                # Store full subsection text
                toc_entries.append((current_section, para, 'sub'))
            story.append(Paragraph(para, subheading_style))
            current_subsection = para
            story.append(Spacer(1, 0.05*inch))
        
        # Check for "TABLE OF CONTENTS" heading - skip it
        elif 'TABLE OF CONTENTS' in para.upper() or 'CONTENTS' in para.upper():
            continue
        
        # Check for title lines (all caps, short, likely a title)
        elif para.isupper() and 10 < len(para) < 100 and not para.startswith('TABLE'):
            # Might be a section title without number
            if not current_section or len(toc_entries) == 0:
                # Use the para as the title, and assign a number
                toc_entries.append((str(len(toc_entries) + 1), para, None))
                current_section = str(len(toc_entries))
            story.append(Spacer(1, 0.15*inch))
            story.append(Paragraph(para, heading_style))
            story.append(Spacer(1, 0.1*inch))
        
        else:
            # Regular paragraph - escape HTML and format
            para = para.replace('&', '&amp;').replace('<', '&lt;').replace('>', '&gt;')
            
            # Split very long paragraphs (might be multiple sentences run together)
            if len(para) > 800:
                # Try to split by sentence endings
                sentences = re.split(r'([.!?]\s+)', para)
                current_para = ""
                for i in range(0, len(sentences), 2):
                    if i + 1 < len(sentences):
                        sentence = sentences[i] + sentences[i+1]
                    else:
                        sentence = sentences[i]
                    
                    if len(current_para + sentence) > 600:
                        if current_para:
                            story.append(Paragraph(current_para.strip(), body_style))
                            story.append(Spacer(1, 0.05*inch))
                        current_para = sentence
                    else:
                        current_para += sentence
                
                if current_para:
                    para = current_para.strip()
            
            # Check for key metrics or important statements (lines with numbers/percentages)
            if re.search(r'\$[\d,]+(\.\d+)?[BMK]?|\d+%|EPS|ROE|ROA|P/E|Revenue|Earnings', para):
                # This might be a key metric - use highlight style
                story.append(Paragraph(para, highlight_style))
            else:
                story.append(Paragraph(para, body_style))
            
            story.append(Spacer(1, 0.08*inch))
    
    # Generate Table of Contents
    toc_story = []
    if toc_entries:
        toc_story.append(Spacer(1, 0.5*inch))
        toc_story.append(Paragraph("TABLE OF CONTENTS", toc_title_style))
        toc_story.append(Spacer(1, 0.3*inch))
        
        # Group TOC entries by section - use Paragraph objects for proper wrapping
        # *** FIX: Use toc_header_style for clean header row ***
        toc_data = [[Paragraph('<b>Section</b>', toc_header_style), 
                     Paragraph('<b>Title</b>', toc_header_style), 
                     Paragraph('<b>Page</b>', toc_header_style)]]
        
        for entry in toc_entries:
            # entry[1] is the full title text
            if entry[2] == 'sub':
                # Subsection - indent with bullet. Remove string slicing to allow clean wrapping.
                title_text = f"&nbsp;&nbsp;&nbsp;&nbsp;• {entry[1]}" 
                toc_data.append(['', Paragraph(title_text, toc_subitem_style), ''])
            else:
                # Main section - bold the number. Remove string slicing to allow clean wrapping.
                title_text = entry[1] 
                toc_data.append([
                    Paragraph(f'<b>{entry[0]}</b>', toc_item_style),
                    Paragraph(title_text, toc_item_style),
                    '' # Page number remains empty here as doc.build has not run yet
                ])
        
        # Create professional TOC table with proper column widths for wrapping
        toc_table = Table(toc_data, colWidths=[0.7*inch, 4.8*inch, 0.7*inch], repeatRows=1)
        toc_table.setStyle(TableStyle([
            # Header row
            ('BACKGROUND', (0, 0), (-1, 0), HexColor('#1a1a1a')),
            ('TEXTCOLOR', (0, 0), (-1, 0), HexColor('#ffffff')),
            ('ALIGN', (0, 0), (-1, 0), 'CENTER'),
            ('ALIGN', (1, 0), (1, 0), 'LEFT'),
            ('FONTNAME', (0, 0), (-1, 0), 'Helvetica-Bold'),
            ('FONTSIZE', (0, 0), (-1, 0), 11),
            ('BOTTOMPADDING', (0, 0), (-1, 0), 12),
            ('TOPPADDING', (0, 0), (-1, 0), 12),
            # Data rows
            ('BACKGROUND', (0, 1), (-1, -1), HexColor('#ffffff')),
            ('TEXTCOLOR', (0, 1), (-1, -1), HexColor('#2c2c2c')),
            ('FONTNAME', (0, 1), (-1, -1), 'Helvetica'),
            ('FONTSIZE', (0, 1), (-1, -1), 10),
            # Section numbers column
            ('FONTSIZE', (0, 1), (0, -1), 10),
            ('FONTNAME', (0, 1), (0, -1), 'Helvetica-Bold'),
            ('ALIGN', (0, 1), (0, -1), 'CENTER'),
            ('VALIGN', (0, 1), (0, -1), 'TOP'),
            # Title column - allow wrapping
            ('ALIGN', (1, 1), (1, -1), 'LEFT'),
            ('VALIGN', (1, 1), (1, -1), 'TOP'),
            # Page number column
            ('ALIGN', (2, 1), (2, -1), 'CENTER'),
            ('VALIGN', (2, 1), (2, -1), 'TOP'),
            # Grid and borders
            ('GRID', (0, 0), (-1, -1), 0.5, HexColor('#e0e0e0')),
            ('LINEBELOW', (0, 0), (-1, 0), 2, HexColor('#ffffff')),
            # Spacing
            ('VALIGN', (0, 0), (-1, 0), 'MIDDLE'),  # Header row
            ('LEFTPADDING', (0, 0), (-1, -1), 8),
            ('RIGHTPADDING', (0, 0), (-1, -1), 8),
            ('TOPPADDING', (0, 1), (-1, -1), 6),
            ('BOTTOMPADDING', (0, 1), (-1, -1), 6),
            # Row height for multi-line content
            ('MINIMUMHEIGHT', (0, 1), (-1, -1), 20),
            # Alternating row colors for subsections
            ('ROWBACKGROUNDS', (0, 1), (-1, -1), [HexColor('#ffffff'), HexColor('#f8f9fa')]),
        ]))
        
        toc_story.append(toc_table)
        toc_story.append(PageBreak())
    
    # Combine: Title page -> TOC -> Content
    story = title_page + toc_story + story
    
    # Debug: Check if we have content
    print(f"   📑 Title page elements: {len(title_page)}")
    print(f"   📑 TOC elements: {len(toc_story)}")
    print(f"   📑 Content elements: {len([x for x in story if x not in title_page and x not in toc_story])}")
    
    if len(story) <= len(title_page) + len(toc_story):
        print("   ⚠️  WARNING: No content found in report! Check if LLM response was properly formatted.")
    
    # --- CRITICAL CHANGE: Build PDF with custom canvas and first_page function ---
    doc.build(story, canvasmaker=NumberedCanvas, onFirstPage=first_page, onLaterPages=first_page)
    # --- END OF CRITICAL CHANGE ---

# Debug: Check response length
print(f"\n📊 Report content length: {len(result.response)} characters")
print(f"📄 Number of paragraphs (by double newline): {len(result.response.split(chr(10)+chr(10)))}")

# Generate PDF
print("\n📝 Generating PDF...")
make_pdf(result.response)
print("✅ Financial report saved to financial_report.pdf")